function [c_p,f_p,new_geo_p,figure_counter,c_p_H,f_p_H,...
    w_p_Man,A_p_Man,P_p_Man,R_p_Man,V_p_Man,Q_p_Man,H_p_Man,stage_p] ...
    = Mannings(xs_geo,n,Hi_slope,Lo_slope,inc,figure_counter,units)
%Manning's Function for cross sectional analysis
% [c,f,new_geo]=Mannings(xs_geo,n,S,inc) finds the values of the
% coefficients c and f from the power function provided by Manning's
% Equation where R=c*Q^f.  The function takes the inputs xs_geo as cross
% sectional geometry, the value of n, deemed the Manning's n, S, the slope
% of the water surface, and inc, the increment for depth calculations for
% use in the general Manning's Equation given by:
%
%                   Q=(k/n)*R^(2/3)*S^(1/2)*A
%
% where R is the hydrolic radius, and A is the area of the cross section.
% This function finds Q as a function of R to fit the power 
% function R=c*Q^f

%% Now we will zero out the values of xs_geo
minimums = min(xs_geo);
xmin = minimums(1,1);
ymin = minimums(1,2);

for i=1:length(xs_geo)
    new_geo_p(i,1) = xs_geo(i,1)-xmin;
    new_geo_p(i,2) = xs_geo(i,2)-ymin;
end

%% Save the X and Y values of the x-section for ease of reference
Xxs = new_geo_p(:,1);
Yxs = new_geo_p(:,2);

%% Save the values of the new x and y maximums and minimums
xmin_new = min(Xxs);
ymin_new = min(Yxs);

xmax_new = max(Xxs);
ymax_new = max(Yxs);

%% get the value of y_max_new based on the cutoff options
[ymax_new,cutoff_choice] = Mannings_cutoff(units,figure_counter,new_geo_p);

%% Now the width at each depth increment must be found
for i=1:floor(ymax_new/inc)
    [Xo, Yo]=mminvinterp(Xxs,Yxs,inc*i);
    
    %% This if statement accounts for the assumption that if the banks
    %% do not match, the side is taken as straight
    if mod(length(Xo),2) ~= 0 && cutoff_choice ~= 2
        
        if find(Yxs == ymax_new) <= (length(Xxs)/3)
            Xo(end+1) = xmax_new;
            Yo(end+1) = inc*i;
        elseif find(Yxs == ymax_new) >= (length(Xxs)/3)
            Xo = [0; Xo];
            Yo = [inc*i; Yo];
        end %inner if

    end %outer if
    
    figure(figure_counter)    
    plot(new_geo_p(:,1),new_geo_p(:,2),'g',Xo,Yo,'ro')
    hold on
    legend('Cross Sectional Geometry','Stage Increments',0)
    title('Data for the calculation of the Manning''s Power Function')
    if units == 1
        xlabel('X distance (m)')
        ylabel('Y distance (m)')
    else
        xlabel('X distance (ft)')
        ylabel('Y distance (ft)')
    end

    %% This loop gets the width at each increment by summing the
    %% invidual differences in the Xo vector.
    w_p_Man(i)=0;
    for j=length(Xo):-2:2
        w_p_Man(i) = Xo(j) - Xo(j-1) + w_p_Man(i);
    end
    
    %% This loop will get the area under each depth increment and find the
    %% hydrolic radius at each increment
    A_under_curve(i) = 0;
    P_p_Man(i) = 0; 
    for j=2:2:length(Xo)
        
        xo_start = min(find(Xxs>Xo(j-1)));
        xo_end   = max(find(Xxs<Xo(j)));
        
        x_temp(1) = Xo(j-1);
        x_temp(2) = Xxs(xo_start);
        y_temp(1) = Yo(j-1);
        y_temp(2) = Yxs(xo_start);
        
        p = xo_end - xo_start + 2;
        
        x_temp(p) = Xxs(xo_end);
        x_temp(p+1) = Xo(j);
        
        y_temp(p) = Yxs(xo_end);
        y_temp(p+1) = Yo(j);
        
        m=3;
        for k=(xo_start+1):1:(xo_end-1)
            x_temp(m) = Xxs(k);
            y_temp(m) = Yxs(k);
            m=m+1;
        end
        
        %% Calculating the areas for each segment of the increment
        A_under_curve(i) = trapz(x_temp,y_temp) + A_under_curve(i);
        
        %% Calculating the perimeter, P, at each segment of the
        %% increment
        for k=1:(length(x_temp)-1)
            P_p_Man(i) = sqrt((x_temp(k+1) - x_temp(k))^2 +...
                (y_temp(k+1) - y_temp(k))^2) + P_p_Man(i);
        end
        
        %% Must clear x and y temp so no old values remain for next j
        %% iteration
        clear x_temp y_temp
    end
    
    %% Finally, the area of each section is the height increment multiplied
    %% by the width subtracted by the area under the curve
    A_p_Man(i) = w_p_Man(i)*inc*i - A_under_curve(i);
    
    R_p_Man(i) = A_p_Man(i)/P_p_Man(i);
    
    H_p_Man(i) = A_p_Man(i)/w_p_Man(i);
    
    %% Must clear Xo and Yo so no old values remain for next i iteration
    clear Xo Yo
    
    %% Slope interpolation at each increment
    S_p(i) = Lo_slope + ((Hi_slope-Lo_slope)/(floor(ymax_new/inc)-1))*(i-1);
    
    if units == 1
        k = 1.0; % for metric units
    else
        k = 1.486;  % for english units
    end

    %% Finally, all values are present to solve the Manning's Equation
    Q_p_Man(i) = (k/n)*R_p_Man(i)^(2/3)*S_p(i)^(1/2)*A_p_Man(i);
    V_p_Man(i) = Q_p_Man(i)/A_p_Man(i);
    
end

%% Now, finally, the values of c and f can be obtained from the
% relationship R = cQ^f.  In order to do so, it must be broken down into
% the y = mx + b form using logorithmic relations.  Therefore,
% log10(h) = f*log(Q) + log(c)

%find c and f using the hydrolic radius
R = log10(R_p_Man);
Q = log10(Q_p_Man);

p1 = polyfit(Q,R,1);

f_p = p1(1);
c_p = 10^(p1(2));

figure_counter = figure_counter + 1;

%find c and f using the hydrolic depth
H = log10(H_p_Man);

p2 = polyfit(Q,H,1);

f_p_H = p1(1);
c_p_H = 10^(p2(2));

%get the stages
stage_p = inc*(1:floor(ymax_new/inc));